/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 2001-2002 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */

#include "nv-misc.h"
#include "os-interface.h"
#include "nv.h"
#include "nv-freebsd.h"


/*
 * The NVIDIA kernel module's malloc identifier, needed for both tracking
 * and actual allocation/freeing purposes. M_NVIDIA is declared elsewhere
 * to make it known to other parts of the kernel module (nv-freebsd.h).
 */

MALLOC_DEFINE(M_NVIDIA, "nvidia", "NVIDIA memory allocations");


#define MAX_ERROR_STRING    256

RM_STATUS NV_API_CALL os_alloc_contig_pages(
    void **address,
    U032 size
)
{
    *address = contigmalloc(size, M_NVIDIA, 0, 0, ~0, PAGE_SIZE, 0);
    return *address ? RM_OK : RM_ERROR;
}

void NV_API_CALL os_free_contig_pages(
    void *address,
    U032 size
)
{
    contigfree(address, size, M_NVIDIA);
}

/*
 * The core resource manager's favorite source of memory, this routine is
 * called from different contexts, including ISRs. This means that it can
 * not be allowed to sleep when memory is low.
 */

RM_STATUS NV_API_CALL os_alloc_mem(
    void **address,
    U032 size
)
{
    /* XXX Fix me? (malloc flags) */
    *address = malloc(size, M_NVIDIA, M_NOWAIT | M_ZERO);
    return *address ? RM_OK : RM_ERROR;
}

void NV_API_CALL os_free_mem(void *address)
{
    free(address, M_NVIDIA);
}

#define NV_MSECS_PER_TICK       (1000 / hz)
#define NV_MSECS_TO_TICKS(ms)   ((ms) * hz / 1000)
#define NV_USECS_PER_TICK       (1000000 / hz)
#define NV_USECS_TO_TICKS(us)   ((us) * hz / 1000000)

RM_STATUS NV_API_CALL os_delay(U032 MilliSeconds)
{
    unsigned long MicroSeconds;
    unsigned long ticks;
    struct timeval tv_end, tv_aux;

    getmicrotime(&tv_aux);

    if (__NV_ITHREAD() && (MilliSeconds > NV_MAX_ISR_DELAY_MS))
        return RM_ERROR;

    if (__NV_ITHREAD()) {
        DELAY(MilliSeconds * 1000);
        return RM_OK;
    }

    MicroSeconds = MilliSeconds * 1000;
    tv_end.tv_usec = MicroSeconds;
    tv_end.tv_sec = 0;
    /* tv_end = tv_aux + tv_end */
    NV_TIMERADD(&tv_aux, &tv_end, &tv_end);

    ticks = NV_USECS_TO_TICKS(MicroSeconds);

    if (ticks > 0) {
        do {
            tsleep((void *)os_delay, PUSER | PCATCH, "delay", ticks);
            getmicrotime(&tv_aux);
            if (NV_TIMERCMP(&tv_aux, &tv_end, <)) {
                /* tv_aux = tv_end - tv_aux */
                NV_TIMERSUB(&tv_end, &tv_aux, &tv_aux);
                MicroSeconds = tv_aux.tv_usec + (tv_aux.tv_sec * 1000000);
            } else
                MicroSeconds = 0;
        } while ((ticks = NV_USECS_TO_TICKS(MicroSeconds)) > 0);
    }

    if (MicroSeconds > 0)
        DELAY(MicroSeconds);

    return RM_OK;
}

RM_STATUS NV_API_CALL os_delay_us(U032 MicroSeconds)
{
    if (__NV_ITHREAD() && (MicroSeconds > NV_MAX_ISR_DELAY_US))
        return RM_ERROR;
    DELAY(MicroSeconds);
    return RM_OK;
}

U032 NV_API_CALL os_get_cpu_frequency(void)
{
    return ((tsc_freq + 4999) / 1000000);
}

U032 NV_API_CALL os_get_current_process(void)
{
    return curproc->p_pid;
}

RM_STATUS NV_API_CALL os_get_current_time(
    U032 *sec,
    U032 *usec
)
{
    struct timeval tv;

    getmicrotime(&tv);

    *sec  = tv.tv_sec;
    *usec = tv.tv_usec;

    return RM_OK;
}

BOOL NV_API_CALL os_is_administrator(PHWINFO pDev)
{
    return suser(CURTHREAD) ? FALSE : TRUE;
}

U008 NV_API_CALL os_io_read_byte(
    PHWINFO pDev,
    U032 address
)
{
    /* XXX Fix me? (bus_space access) */
    return inb(address);
}

void NV_API_CALL os_io_write_byte(
    PHWINFO pDev,
    U032 address,
    U008 value
)
{
    /* XXX Fix me? (bus_space access) */
    outb(address, value);
}

U016 NV_API_CALL os_io_read_word(
    PHWINFO pDev,
    U032 address
)
{
    /* XXX Fix me? (bus_space access) */
    return inw(address);
}

void NV_API_CALL os_io_write_word(
    PHWINFO pDev,
    U032 address,
    U016 value
)
{
    /* XXX Fix me? (bus_space access) */
    return outw(address, value);
}

U032 NV_API_CALL os_io_read_dword(
    PHWINFO pDev,
    U032 address
)
{
    /* XXX Fix me? (bus_space access) */
    return inl(address);
}

void NV_API_CALL os_io_write_dword(
    PHWINFO pDev,
    U032 address,
    U032 value
)
{
    /* XXX Fix me? (bus_space access) */
    outl(address, value);
}

void* NV_API_CALL os_map_kernel_space(
    NvU64 start,
    NvU64 size,
    U032 mode
)
{
    int map_mode;

#if defined(NVCPU_X86) && !defined(PAE)
    if (start > 0xffffffff)
        return NULL;
#endif

    if (start & PAGE_MASK)
        return NULL;

    size = NV_ALIGN_UP(size, PAGE_SIZE);

    switch (mode) {
        case NV_MEMORY_CACHED:
            map_mode = PAT_WRITE_BACK;
            break;
        case NV_MEMORY_WRITECOMBINED:
            map_mode = PAT_WRITE_COMBINING;
            break;
        case NV_MEMORY_UNCACHED:
        case NV_MEMORY_DEFAULT:
            map_mode = PAT_UNCACHEABLE;
            break;
        default:
            nv_printf(NV_DBG_ERRORS,
                      "NVRM: unknown mode in os_map_kernel_space()\n");
            return NULL;
    }

    return pmap_mapdev_attr(start, size, map_mode);
}

void NV_API_CALL os_unmap_kernel_space(
    void *address,
    NvU64 size
)
{
    pmap_unmapdev_attr((vm_offset_t)address, size);
}

void* NV_API_CALL os_map_kernel_space_high(
    U032 pfn,
    U032 size
)
{
    U032 start;
    if (!(pfn & ~0xfffff)) {
        start = pfn << PAGE_SHIFT;
        return os_map_kernel_space(start, size, NV_MEMORY_CACHED);
    }
    return NULL;
}

void NV_API_CALL os_unmap_kernel_space_high(
    void *addr,
    U032 pfn,
    U032 size
)
{
    os_unmap_kernel_space(addr, size);
}

RM_STATUS NV_API_CALL os_set_mem_range(
    U032 start,
    U032 size,
    U032 mode
)
{
    int arg;
    struct mem_range_desc mrd;

    mrd.mr_base  = start;
    mrd.mr_len   = size;
    mrd.mr_flags = MDF_WRITECOMBINE;

    strcpy(mrd.mr_owner, "NVIDIA");
    arg = MEMRANGE_SET_UPDATE;

    if (mem_range_attr_set(&mrd, &arg))
        return RM_ERROR;

    return RM_OK;
}

RM_STATUS NV_API_CALL os_unset_mem_range(
    U032 start,
    U032 size
)
{
    int arg;
    struct mem_range_desc mrd;

    mrd.mr_base = start;
    mrd.mr_len  = size;
    arg = MEMRANGE_SET_REMOVE;

    if (mem_range_attr_set(&mrd, &arg))
        return RM_ERROR;

    return RM_OK;
}


/*
 * The current debug level is used to determine if certain debug messages
 * are printed to the system console/log files or not. It defaults to the
 * highest debug level, i.e. the lowest debug output.
 */

U032 cur_debuglevel = 0xffffffff;

void NV_API_CALL os_dbg_init(void)
{
    U032 new_debuglevel;
    nv_stack_t *sp;

    NV_UMA_ZONE_ALLOC_STACK(sp);
    if (sp == NULL)
        return;

    if (rm_read_registry_dword(sp, NULL, "NVreg", "ResmanDebugLevel",
            &new_debuglevel) == RM_OK) {
        if (new_debuglevel != 0xffffffff)
            cur_debuglevel = new_debuglevel;
    }

    NV_UMA_ZONE_FREE_STACK(sp);
}

RM_STATUS NV_API_CALL os_schedule(void)
{
    return RM_ERR_NOT_SUPPORTED;
}

void NV_API_CALL os_dbg_set_level(U032 new_debuglevel)
{
    cur_debuglevel = new_debuglevel;
}

void NV_API_CALL os_dbg_breakpoint(void)
{
#ifdef DEBUG
    /* FreeBSD 5.x kdb */
    kdb_enter("os_dbg_breakpoint()");
#endif
}

void NV_API_CALL out_string(const char *message)
{
#if defined(DEBUG) || defined(QA_BUILD)
    printf("%s", message);
#endif
}

static char nv_error_string[MAX_ERROR_STRING];

int NV_API_CALL nv_printf(
    U032 debuglevel,
    const char *format,
    ...
)
{
    char *message = nv_error_string;
    va_list arglist;
    int chars_written = 0;

    if (debuglevel >= ((cur_debuglevel >> 4) & 3)) {
        va_start(arglist, format);
        chars_written = vsprintf(message, format, arglist);
        va_end(arglist);
        printf("%s", message);
    }

    return chars_written;
}

int NV_API_CALL nv_snprintf(
    char *buf,
    unsigned int size,
    const char *fmt,
    ...
)
{
    va_list arglist;
    int chars_written;

    va_start(arglist, fmt);
    chars_written = vsnprintf(buf, size, fmt, arglist);
    va_end(arglist);

    return chars_written;
}

void NV_API_CALL nv_os_log(
    int loglevel,
    const char *fmt,
    void *ap
)
{
    int l;
    sprintf(nv_error_string, "NVRM: ");
    l = strlen(nv_error_string);
    vsnprintf(nv_error_string + l, MAX_ERROR_STRING - l, fmt, ap);
    printf("%s", nv_error_string);
}

S032 NV_API_CALL os_mem_cmp(
    const U008 *buf0,
    const U008 *buf1,
    U032 length
)
{
    return memcmp(buf0, buf1, length);
}

U008* NV_API_CALL os_mem_copy(
    U008 *dst,
    const U008 *src,
    U032 length
)
{
#if defined(NVCPU_X86_64)
    uint32_t i;
    for (i = 0; i < length; i++) dst[i] = src[i];
    return dst;
#else
    return memcpy(dst, src, length);
#endif
}

RM_STATUS NV_API_CALL os_memcpy_from_user(
    void *dst,
    const void *src,
    U032 length
)
{
    if (src < (void *) VM_MAXUSER_ADDRESS)
        return copyin(src, dst, length)  ? RM_ERR_INVALID_POINTER : RM_OK;

    return os_mem_copy(dst, src, length) ? RM_ERR_INVALID_POINTER : RM_OK;
}

RM_STATUS NV_API_CALL os_memcpy_to_user(
    void *dst,
    const void *src,
    U032 length
)
{
    if (dst < (void *) VM_MAXUSER_ADDRESS)
        return copyout(src, dst, length) ? RM_ERR_INVALID_POINTER : RM_OK;

    return os_mem_copy(dst, src, length) ? RM_ERR_INVALID_POINTER : RM_OK;
}

void* NV_API_CALL os_mem_set(
    void *b,
    U008 c,
    U032 length
)
{
    return memset(b, c, length);
}

S032 NV_API_CALL os_string_compare(
    const char *s1,
    const char *s2
)
{
    return strcmp(s1, s2);
}

char* NV_API_CALL os_string_copy(
    char *dst,
    const char *src
)
{
    return strcpy(dst, src);
}

U032 NV_API_CALL os_string_length(const char* s)
{
    return strlen(s);
}

RM_STATUS NV_API_CALL os_strncpy_from_user(
    char *dst,
    const char *src,
    U032 n
)
{
    return copyinstr(src, dst, n, NULL) ? RM_ERR_INVALID_POINTER : RM_OK;
}

U032 NV_API_CALL os_get_page_size(void)
{
    return PAGE_SIZE;
}

NvU64 NV_API_CALL os_get_page_mask(void)
{
    /*
     * On FreeBSD, PAGE_MASK means (PAGE_SIZE - 1); on Linux it means the
     * opposite, ~(PAGE_SIZE - 1); that is what this function is expected
     * to return.
     */
    return ~PAGE_MASK;
}

NvU64 NV_API_CALL os_get_system_memory_size(void)
{
    return ((NvU64)physmem * PAGE_SIZE) / RM_PAGE_SIZE;
}

U032 NV_API_CALL os_get_cpu_count(void)
{
    return mp_ncpus;
}

RM_STATUS NV_API_CALL os_flush_cpu_cache(void)
{
    /*
     * XXX This will do for now, but this may need to be extended
     * to make IPI calls (flushing all caches).
     */
    __asm__ __volatile__("wbinvd": : :"memory");
    return RM_OK;
}

void NV_API_CALL os_flush_cpu_write_combine_buffer(void)
{
    __asm__ __volatile__("sfence": : :"memory");
}

RM_STATUS NV_API_CALL os_raise_smp_barrier(void)
{
    return RM_ERR_NOT_SUPPORTED;
}

RM_STATUS NV_API_CALL os_clear_smp_barrier(void)
{
    return RM_ERR_NOT_SUPPORTED;
}

struct os_mutex {
    nv_stack_t *sp;
    struct mtx mutex_mtx;
    struct cv mutex_cv;
    int refcnt;
};

RM_STATUS NV_API_CALL os_alloc_sema(void **semaphore)
{
    nv_stack_t *sp;
    struct os_mutex *mtx;
    RM_STATUS status;

    NV_UMA_ZONE_ALLOC_STACK(sp);
    if (sp == NULL)
        return RM_ERR_NO_FREE_MEM;

    status = os_alloc_mem((void **)&mtx, sizeof(struct os_mutex));
    if (status != RM_OK) {
        NV_UMA_ZONE_FREE_STACK(sp);
        return status;
    }

    mtx_init(&mtx->mutex_mtx, "os.mutex_mtx", NULL, MTX_DEF | MTX_RECURSE);
    cv_init(&mtx->mutex_cv, "os.mutex_cv");

    mtx->sp = sp;
    mtx->refcnt = 1;

    *semaphore = (void *) mtx;

    return RM_OK;
}

RM_STATUS NV_API_CALL os_free_sema(void *semaphore)
{
    struct os_mutex *mtx = semaphore;
    nv_stack_t *sp;

    sp = mtx->sp;
    NV_UMA_ZONE_FREE_STACK(sp);

    mtx_destroy(&mtx->mutex_mtx);
    cv_destroy(&mtx->mutex_cv);

    os_free_mem(semaphore);

    return RM_OK;
}

RM_STATUS NV_API_CALL os_acquire_sema(void *semaphore)
{
    struct os_mutex *mtx = semaphore;

    mtx_lock(&mtx->mutex_mtx);
    if (mtx->refcnt > 0)
        rm_disable_interrupts(mtx->sp);
    mtx->refcnt--;
    if (mtx->refcnt < 0)
        cv_wait(&mtx->mutex_cv, &mtx->mutex_mtx);
    mtx_unlock(&mtx->mutex_mtx);

    return RM_OK;
}

BOOL NV_API_CALL os_cond_acquire_sema(void *semaphore)
{
    struct os_mutex *mtx = semaphore;

    mtx_lock(&mtx->mutex_mtx);
    if (mtx->refcnt < 1) {
        mtx_unlock(&mtx->mutex_mtx);
        return FALSE;
    } else {
        rm_disable_interrupts(mtx->sp);
        mtx->refcnt--;
        mtx_unlock(&mtx->mutex_mtx);
    }

    return TRUE;
}

RM_STATUS NV_API_CALL os_release_sema(void *semaphore)
{
    struct os_mutex *mtx = semaphore;

    mtx_lock(&mtx->mutex_mtx);
    if (mtx->refcnt < 0)
        cv_signal(&mtx->mutex_cv);
    else
        rm_enable_interrupts(mtx->sp);
    mtx->refcnt++;
    mtx_unlock(&mtx->mutex_mtx);

    return RM_OK;
}

BOOL NV_API_CALL os_is_acquired_sema(void *semaphore)
{
    struct os_mutex *mtx = semaphore;
    return (mtx->refcnt < 1);
}

BOOL NV_API_CALL os_pat_supported(void)
{
    /*
     * FreeBSD has no native PAT support and there's no good
     * way to implement it privately as we do on Linux.
     */
    return FALSE;
}

RM_STATUS NV_API_CALL os_set_mlock_capability()
{
    return (RM_ERROR);
}

S032 NV_API_CALL os_mlock_user_memory(
    void *address,
    U032 length
)
{
    return -1;
}

S032 NV_API_CALL os_munlock_user_memory(
    void *address,
    U032 length
)
{
    return -1;
}

RM_STATUS NV_API_CALL os_check_process_map_limit(
    NvU64 proc_max_map_count
)
{
    return (RM_ERROR);
}

void NV_API_CALL os_register_compatible_ioctl(
    U032 cmd,
    U032 size
)
{
}

void NV_API_CALL os_unregister_compatible_ioctl(
    U032 cmd,
    U032 size
)
{
}

RM_STATUS NV_API_CALL os_disable_console_access(void)
{
    return RM_OK;
}

RM_STATUS NV_API_CALL os_enable_console_access(void)
{
    return RM_OK;
}
